//---------------------------------
// 2020/12/5

#include "WebPageHandler.h"

#include "INA219I2C.h"
#include "Storage.h"
#include "StrList.h"
#include "TimelyTaskManager.h"

// SolarCharger.ino の変数を使うための宣言
extern INA219I2C ina219;
extern TimelyTaskManager timelyTaskManager;
extern bool logMode;
extern bool load;
extern bool loadCut;
extern bool sleepMode;
extern String logDir;

extern float vBatTarget;
extern float cBatTarget;
extern float vLoadCut;
extern float vLoadRecover;

// SolarCharger.ino の関数を使うためのプロトタイプ宣言
String vchargeStr();
void startLog();
void stopLog();
void loadControl();
void sleepControl();

// Web Server
httpd_handle_t WebPageHandler::server = NULL;

// サーバーで処理するページ、URI
const char WebPageHandler::pageBattery[] = "/battery.php";
const char WebPageHandler::pageLogFile[] = "/logfile.php";
const char WebPageHandler::pageSystem[] = "/system.php";
const char WebPageHandler::pageDownload[] = "/download.php";

const httpd_uri_t WebPageHandler::uriBattery = {.uri = pageBattery, .method = HTTP_GET, .handler = handleBatteryPage, .user_ctx = NULL};
const httpd_uri_t WebPageHandler::uriLogFile = {.uri = pageLogFile, .method = HTTP_GET, .handler = handleLogFilePage, .user_ctx = NULL};
const httpd_uri_t WebPageHandler::uriSystem = {.uri = pageSystem, .method = HTTP_GET, .handler = handleSystemPage, .user_ctx = NULL};
const httpd_uri_t WebPageHandler::uriDownload = {.uri = pageDownload, .method = HTTP_GET, .handler = handleDownload, .user_ctx = NULL};

// Callback
std::function<void(uint8_t)> WebPageHandler::callBack;

// HTTP GETのパラメーターを取得
String WebPageHandler::getReqArg(httpd_req_t *req, String arg) {
  char query[256];
  char param[64];
  uint32_t len;
  len = httpd_req_get_url_query_len(req) + 1;
  if (len > 1) {
    if (httpd_req_get_url_query_str(req, query, len) == ESP_OK) {
      if (httpd_query_key_value(query, arg.c_str(), param, sizeof(param)) == ESP_OK) {
        return String(param);
      }
    }
  }
  return "";
}

// サーバーを開始、初期設定
void WebPageHandler::startServer(std::function<void(uint8_t)> f) {
  callBack = f;

  httpd_config_t config = HTTPD_DEFAULT_CONFIG();
  if (httpd_start(&server, &config) == ESP_OK) {
    httpd_register_uri_handler(server, &uriBattery);
    httpd_register_uri_handler(server, &uriLogFile);
    httpd_register_uri_handler(server, &uriSystem);
    httpd_register_uri_handler(server, &uriDownload);
  }
}

// 共通のHTMLヘッダーを返す
String WebPageHandler::getHeader() {
  return String() +
         "<head>\n"
         "  <meta charset=\"utf-8\">\n"
         "  <meta http-equiv=\"X-UA-Compatible\" content=\"IE=edge\">\n"
         "  <meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">\n"
         "  <title>E32-SolarCharger</title>\n\n"

         "  <!-- Bootswatch flatly -->\n"
         "  <link "
         "href=\"https://maxcdn.bootstrapcdn.com/bootswatch/3.3.7/flatly/"
         "bootstrap.min.css\"rel=\"stylesheet\"integrity=\"sha384-+ENW/"
         "yibaokMnme+vBLnHMphUYxHs34h9lpdbSLuAwGkOKFRl4C34WkjazBtb7eT\"crossorigin=\"anonymous\">\n"
         "</head>\n";
}

// 共通のナビゲーションバーを返す
String WebPageHandler::getNavBar(const char *page) {
  String resp = "";
  resp +=
      "<nav class=\"navbar navbar-default navbar-fixed-top\">\n"
      "  <div class=\"container-fluid\">\n"
      "    <div class=\"navbar-header\">\n"
      "      <button type=\"button\" class=\"navbar-toggle collapsed\" "
      "data-toggle=\"collapse\"data-target=\"#bs-example-navbar-collapse-1\">\n"
      "        <span class=\"sr-only\">Toggle navigation</span>\n"
      "        <span class=\"icon-bar\"></span>\n"
      "        <span class=\"icon-bar\"></span>\n"
      "        <span class=\"icon-bar\"></span>\n"
      "      </button>\n"
      "      <a class=\"navbar-brand\" href=\"#\">E32-SolarCharger</a>\n"
      "    </div>\n\n"
      "    <div class=\"collapse navbar-collapse\" id=\"bs-example-navbar-collapse-1\">\n"
      "      <ul class=\"nav navbar-nav\">\n";

  if (0 == strcmp(page, pageBattery)) {
    resp += "        <li class=\"active\"><a href=\"" + String(pageBattery) + "\">Battery<span class=\"sr-only\">(current)</span></a></li>\n";
  } else {
    resp += "        <li><a href=\"" + String(pageBattery) + "\">Battery</a></li>\n";
  }
  if (0 == strcmp(page, pageLogFile)) {
    resp += "        <li class=\"active\"><a href=\"" + String(pageLogFile) + "\">Log File<span class=\"sr-only\">(current)</span></a></li>\n";
  } else {
    resp += "        <li><a href=\"" + String(pageLogFile) + "\">Log File</a></li>\n";
  }
  if (0 == strcmp(page, pageSystem)) {
    resp += "        <li class=\"active\"><a href=\"" + String(pageSystem) + "\">System<span class=\"sr-only\">(current)</span></a></li>\n";
  } else {
    resp += "        <li><a href=\"" + String(pageSystem) + "\">System</a></li>\n";
  }
  resp +=
      "      </ul>\n"
      "    </div>\n"
      "  </div>\n"
      "</nav>\n\n";
  return resp;
}

// 共通のBootstrapスクリプトを返す
String WebPageHandler::getScript() {
  String resp =
      "<script src=\"https://ajax.googleapis.com/ajax/libs/jquery/1.12.4/jquery.min.js\"></script>\n"
      "<script src=\"https://maxcdn.bootstrapcdn.com/bootstrap/3.3.7/js/bootstrap.min.js\" integrity=\"sha384-Tc5IQib027qvyjSMfHjOMaLkfuWVxZxUPnCJA7l2mCWNIpG9mGCD8wGNIcPD7Txa\" "
      "crossorigin=\"anonymous\"></script>\n";
  return resp;
}

// バッテリー情報ページのハンドラー
esp_err_t WebPageHandler::handleBatteryPage(httpd_req_t *req) {
  String resp = "";

  String actionStr = getReqArg(req, "action");
  if (actionStr == "startLog") {
    startLog();
  } else if (actionStr == "stopLog") {
    stopLog();
  } else if (actionStr == "sleepAuto") {
    sleepMode = true;  // Sleep control() is after responding HTTP request
  } else if (actionStr == "sleepOff") {
    sleepMode = false;
  } else if (actionStr == "loadAuto") {
    load = true;
    loadControl();
    delay(100);
  } else if (actionStr == "loadOff") {
    load = false;
    loadControl();
    delay(100);
  }

  resp +=
      "<!DOCTYPE html>\n\n"
      "<html lang=\"ja\">\n\n";

  resp += getHeader();
  resp += "<body style=\"padding-top:70px\">\n\n";

  resp += getNavBar(pageBattery);

  resp +=
      "<div class=\"container\">\n"
      "  <div class=\"row\"><div class=\"col-sm-10 col-md-8\">\n"
      "    <div class=\"panel panel-primary\">\n"
      "      <div class=\"panel-heading\">\n"
      "        <h3 class=\"panel-title\">Battery Status</h3>\n"
      "      </div>\n"
      "      <div class=\"panel-body\">\n";
  char p[8];
  resp = resp + "        <h4>Max Charge Voltage : " + dtostrf(vBatTarget, 1, 1, p) + "V</h4>\n";
  resp = resp + "        <h4>Max Charge Current : " + dtostrf(cBatTarget, 1, 1, p) + "A</h4>\n";
  resp = resp + "        <h4>Load Cut Voltage : " + dtostrf(vLoadCut, 1, 1, p) + "V</h4>\n";
  resp = resp + "        <h4>Load Recovery Voltage : " + dtostrf(vLoadRecover, 1, 1, p) + "V</h4>\n";
  resp = resp + "        <h4>Battery Voltage : " + ina219.vbusStr(false) + "V</h4>\n";
  resp = resp + "        <h4>Battery Current : " + ina219.currentStr(false) + "A</h4>\n";
  resp = resp + "        <h4>Battery Power : " + dtostrf(ina219.vbus(false) * ina219.current(false), 1, 2, p) + "W</h4>\n";
  resp = resp + "        <h4>Input Voltage : " + vchargeStr() + "V</h4>\n";

  if (logMode) {
    resp += "        <h4>Log : Running\n";
  } else {
    resp += "        <h4>Log : Stop\n";
  }
  if (sleepMode) {
    resp += "        <h4>Sleep : Auto\n";
  } else {
    resp += "        <h4>Sleep : Off\n";
  }
  if (load) {
    resp += "        <h4>Load : Auto\n";
  } else {
    resp += "        <h4>Load : Off\n";
  }
  resp +=
      "      </div>\n"
      "    </div>\n";
  resp +=
      "    <form method=\"GET\" class=\"form-horizontal\">\n"
      "      <div class=\"form-group\">\n"
      "        <div class=\"col-xs-6\">\n"
      "          <button type=\"submit\" name=\"action\" value=\"startLog\" class=\"btn btn-default btn-block\">Start Recording</button>\n"
      "        </div>\n"
      "        <div class=\"col-xs-6\">\n"
      "          <button type=\"submit\" name=\"action\"value=\"stopLog\" class=\"btn btn-default btn-block\">Stop Recording</button>\n"
      "        </div>\n"
      "      </div>\n"
      "      <div class=\"form-group\">\n"
      "        <div class=\"col-xs-6\">\n"
      "          <button type=\"submit\" name=\"action\" value=\"sleepAuto\" class=\"btn btn-default btn-block\">Sleep Auto</button>\n"
      "        </div>\n"
      "        <div class=\"col-xs-6\">\n"
      "          <button type=\"submit\" name=\"action\"value=\"sleepOff\" class=\"btn btn-default btn-block\">Sleep OFF</button>\n"
      "        </div>\n"
      "      </div>\n"
      "      <div class=\"form-group\">\n"
      "        <div class=\"col-xs-6\">\n"
      "          <button type=\"submit\" name=\"action\" value=\"loadAuto\" class=\"btn btn-default btn-block\">Load Auto</button>\n"
      "        </div>\n"
      "        <div class=\"col-xs-6\">\n"
      "          <button type=\"submit\" name=\"action\" value=\"loadOff\" class=\"btn btn-default btn-block\">Load OFF</button>\n"
      "        </div>\n"
      "      </div>\n"
      "    </form>\n"

      "  </div></div><!-- row, col -->\n"
      "</div>\n\n";

  resp += getScript();

  resp +=
      "</body>\n"
      "</html>\n";

  httpd_resp_set_type(req, "text/html");
  httpd_resp_send(req, resp.c_str(), strlen(resp.c_str()));

  if (actionStr == "sleepAuto") {
    timelyTaskManager.setOneTimeTask(idSleepControlOneTime, sleepControl);
  }
  return ESP_OK;
}

// ログファイルページのハンドラー
esp_err_t WebPageHandler::handleLogFilePage(httpd_req_t *req) {
  String resp = "";

  String selectFile = "";
  String filelistStr = getReqArg(req, "filelist");
  String actionStr = getReqArg(req, "action");

  if (actionStr == "download") {
    Storage::opLog(timelyTaskManager.getTimeStr() + " Download Log File " + filelistStr + "\n");
    String path = logDir + "/" + filelistStr;
    downloadFile(req, path);
    return ESP_OK;
  } else if (actionStr == "delete") {
    Storage::opLog(timelyTaskManager.getTimeStr() + " Delete Log File " + filelistStr + "\n");
    String path = logDir + "/" + filelistStr;
    Storage::remove(path);
    Storage::end();
  } else if (actionStr == "deleteall") {
    Storage::opLog(timelyTaskManager.getTimeStr() + " Delete All Log Files " + filelistStr + "\n");
    File root = Storage::open(logDir, FILE_READ);
    if (root) {
      if (root.isDirectory()) {
        File file = root.openNextFile();
        while (file) {
          String path = file.name();
          file.close();
          Storage::remove(path);
          file = root.openNextFile();
        }
      }
    }
    Storage::end();
  }

  resp +=
      "<!DOCTYPE html>\n\n"
      "<html lang=\"ja\">\n\n";
  resp += getHeader();
  resp += "<body style=\"padding-top:70px\">\n\n";
  resp += getNavBar(pageLogFile);

  resp +=
      "<div class=\"container\">\n"
      "  <div class=\"row\"><div class=\"col-sm-10 col-md-8\">\n";

  resp +=
      "    <form method=\"GET\" class=\"form-horizontal\">\n"
      "      <div class=\"form-group\"><div class=\"col-xs-12\">\n"
      "        <select class=\"form-control\" size=\"8\" name=\"filelist\">\n";

  // Read directory
  uint16_t fileCount = 0;
  StrList *strList = new StrList();

  File root = Storage::open(logDir, FILE_READ);
  if (root) {
    if (root.isDirectory()) {
      File file = root.openNextFile();
      while (file) {
        String fname = file.name();
        fname.replace(logDir + "/", "");
        strList->add(fname);
        fileCount++;
        file.close();
        file = root.openNextFile();
      }
    }
    root.close();
  }
  Storage::end();

  StrListItem *strListItem = strList->getFirst();

  for (uint16_t i = 0; i < fileCount; i++) {
    resp += "          <option>" + strListItem->str + "</option>\n";
    strListItem = strList->getNext();
  }
  delete strList;

  resp +=
      "        </select>\n"
      "      </div></div>\n"
      "      <div class=\"form-group\">\n"
      "        <div class=\"col-xs-6\">\n"
      "          <button type=\"submit\" name=\"action\" value=\"download\" class=\"btn btn-default btn-block\">Download</button>\n"
      "        </div>\n"
      "        <div class=\"col-xs-6\">\n"
      "          <button type=\"submit\" name=\"action\" value=\"delete\" class=\"btn btn-default btn-block\">Delete</button>\n"
      "        </div>\n"
      "      </div>\n"
      "      <div class=\"form-group\">\n"
      "        <div class=\"col-xs-6\">\n"
      "          <button type=\"button\" class=\"btn btn-default btn-block\" data-toggle=\"modal\" data-target=\"#deleteAllDialog\">Delete All</button>\n"
      "        </div>\n"
      "      </div>\n"
      "    </form>\n";

  resp +=
      "  </div></div><!-- row, col -->\n"
      "</div>\n";

  resp +=
      "<div class=\"modal\" id=\"deleteAllDialog\">\n"
      "  <div class=\"modal-dialog\">\n"
      "    <div class=\"modal-content\">\n"
      "      <div class=\"modal-header\">\n"
      "        <button type=\"button\" class=\"close\" data-dismiss=\"modal\" aria-hidden=\"true\">&times;</button>\n"
      "        <h4 class=\"modal-title\">Delete All Files?</h4>\n"
      "      </div>\n"
      "      <!-- <div class=\"modal-body\">\n"
      "      </div> -->\n"
      "      <div class=\"modal-footer\">\n"
      "        <form method=\"GET\">\n"
      "          <button type=\"button\" class=\"btn btn-default\" data-dismiss=\"modal\">Cancel</button>\n"
      "          <button type=\"submit\" class=\"btn btn-primary\" name=\"action\" value=\"deleteall\">OK</button>\n"
      "        </form>\n"
      "      </div>\n"
      "    </div>\n"
      "  </div>\n"
      "</div>\n\n";

  resp += getScript();

  resp +=
      "</body>\n"
      "</html>\n";

  httpd_resp_set_type(req, "text/html");
  httpd_resp_send(req, resp.c_str(), strlen(resp.c_str()));
  return ESP_OK;
}

// システム情報ページのハンドラー
esp_err_t WebPageHandler::handleSystemPage(httpd_req_t *req) {
  String resp = "";
  String actionStr = getReqArg(req, "action");

  if (actionStr == "formattorage") {
    if (!Storage::opLogSD) {
      SPIFFS.format();
    }
    Storage::opLog(timelyTaskManager.getTimeStr() + " Format Storage\n");
  }

  if (actionStr == "clearOpLog") {
    String path = Storage::getOpLogPath();
    Storage::remove(path);
    Storage::end();
    Storage::opLog(timelyTaskManager.getTimeStr() + " Clear Op Log\n");
  }

  resp +=
      "<!DOCTYPE html>\n\n"
      "<html lang=\"ja\">\n\n";
  resp += getHeader();
  resp += "<body style=\"padding-top:70px\">\n\n";
  resp += getNavBar(pageSystem);

  resp +=
      "<div class=\"container\">\n"
      "  <div class=\"row\"><div class=\"col-sm-10 col-md-8\">\n"
      "    <div class=\"panel panel-primary\">\n"
      "      <div class=\"panel-heading\">\n"
      "        <h3 class=\"panel-title\">System Status</h3>\n"
      "      </div>\n"
      "      <div class=\"panel-body\">\n";
  if (Storage::opLogSD) {
    resp += "        <h4>Storage : SD</h4>\n";
  } else {
    SPIFFS.begin(true);
    resp += "        <h4>Storage : SPIFFS</h4>\n";
    resp += "        <h5>Total : " + String(SPIFFS.totalBytes()) + " Bytes</h5>\n";
    resp += "        <h5>Used : " + String(SPIFFS.usedBytes()) + " Bytes</h5>\n";
  }
  resp += "        <h4>Flash Size : " + String(ESP.getFlashChipSize()) + " Bytes</h4>\n";
  resp += "        <h4>Flash Frequency : " + String(ESP.getFlashChipSpeed() / 1000000) + " MHz</h4>\n";
  resp += "        <h4>Free Heap : " + String(ESP.getFreeHeap()) + " Bytes</h4>\n";
  if (Storage::opLogSerial) {
    resp += "        <h4>Operation Log Serial : Enable</h4>\n";
  } else {
    resp += "        <h4>Operation Log Serial : Disable</h4>\n";
  }
  if (Storage::opLogStorage) {
    resp += "        <h4>Operation Log Storage : Enable</h4>\n";
  } else {
    resp += "        <h4>Operation Log Storage : Disable</h4>\n";
  }

  resp +=
      "      </div>\n"
      "    </div>\n"
      "    <form methos=\"POST\" class=\"form-horizontal\"><fieldset>\n"
      "      <div class=\"form-group\">\n"
      "        <div class=\"col-xs-6\">\n"
      "          <button type=\"button\" class=\"btn btn-default btn-block\" data-toggle=\"modal\" data-target=\"#clearOpLogDialog\">Clear Op Log</button>\n"
      "        </div>\n"
      "        <div class=\"col-xs-6\">\n"
      "          <button type=\"button\" class=\"btn btn-default btn-block\" onclick=\"location.href='download.php?file=";
  resp += Storage::getOpLogPath();

  resp +=
      "'\">Download Op Log</button>\n"
      "        </div>\n"
      "      </div>\n";
  if (!Storage::opLogSD) {
    resp +=
        "      <div class=\"form-group\">\n"
        "        <div class=\"col-xs-6\">\n"
        "          <button type=\"button\" class=\"btn btn-default btn-block\" data-toggle=\"modal\" data-target=\"#formattorageDialog\">Format Storage</button>\n"
        "        </div>\n"
        "      </div>\n";
  }
  resp +=
      "    </fieldset></form>\n"
      "  </div></div><!-- row, col -->\n"
      "</div>\n\n"
      "<div class=\"modal\" id=\"clearOpLogDialog\">\n"
      "  <div class=\"modal-dialog\">\n"
      "    <div class=\"modal-content\">\n"
      "      <div class=\"modal-header\">\n"
      "        <button type=\"button\" class=\"close\" data-dismiss=\"modal\" aria-hidden=\"true\">&times;</button>\n"
      "        <h4 class=\"modal-title\">Clear Op Log file?</h4>\n"
      "      </div>\n"
      "      <!-- <div class=\"modal-body\">\n"
      "      </div> -->\n"
      "      <div class=\"modal-footer\">\n"
      "        <form methos=\"POST\">\n"
      "          <button type=\"button\" class=\"btn btn-default\" data-dismiss=\"modal\">Cancel</button>\n"
      "          <button type=\"submit\" class=\"btn btn-primary\" name=\"action\" value=\"clearOpLog\">OK</button>\n"
      "        </form>\n"
      "      </div>\n"
      "    </div>\n"
      "  </div>\n"
      "</div>\n\n"
      "<div class=\"modal\" id=\"formattorageDialog\">\n"
      "  <div class=\"modal-dialog\">\n"
      "    <div class=\"modal-content\">\n"
      "      <div class=\"modal-header\">\n"
      "        <button type=\"button\" class=\"close\" data-dismiss=\"modal\" aria-hidden=\"true\">&times;</button>\n"
      "        <h4 class=\"modal-title\">Format Storage?</h4>\n"
      "      </div>\n"
      "      <!-- <div class=\"modal-body\">\n"
      "      </div> -->\n"
      "      <div class=\"modal-footer\">\n"
      "        <form methos=\"POST\">\n"
      "          <button type=\"button\" class=\"btn btn-default\" data-dismiss=\"modal\">Cancel</button>\n"
      "          <button type=\"submit\" class=\"btn btn-primary\" name=\"action\" value=\"formattorage\">OK</button>\n"
      "        </form>\n"
      "      </div>\n"
      "    </div>\n"
      "  </div>\n"
      "</div>\n\n";

  resp += getScript();

  resp +=
      "</body>\n"
      "</html>\n";

  httpd_resp_set_type(req, "text/html");
  httpd_resp_send(req, resp.c_str(), strlen(resp.c_str()));
  return ESP_OK;
}

// ダウンロードアドレスのハンドラー
esp_err_t WebPageHandler::handleDownload(httpd_req_t *req) {
  String path = getReqArg(req, "file");
  downloadFile(req, path);
  return ESP_OK;
}

// 指定したパスのファイルをダウンロードする
#define DL_BUF_SIZE 128
void WebPageHandler::downloadFile(httpd_req_t *req, String path) {
  // パスからファイル名だけを取り出す
  String fname = path;
  while (fname.indexOf("/") != -1) {
    fname = fname.substring(fname.indexOf("/") + 1);
  }
  String fnameHeader = "attachment; filename=\"" + fname + "\"";
  httpd_resp_set_hdr(req, "Content-Disposition", fnameHeader.c_str());

  // 拡張子を取り出す
  String ext = fname;
  while (ext.indexOf(".") != -1) {
    ext = ext.substring(ext.indexOf(".") + 1);
  }

  if (ext.equals("txt") || ext.equals("log") || ext.equals("csv")) {
    httpd_resp_set_type(req, "text/plain");
  } else {
    httpd_resp_set_type(req, "application/octet-stream");
  }

  char buf[DL_BUF_SIZE];
  int readSize = 0;

  File file = Storage::open(path, FILE_READ);
  if (file) {
    while (1) {
      readSize = file.readBytes(buf, DL_BUF_SIZE);
      if (readSize == 0) break;
      httpd_resp_send_chunk(req, buf, readSize);
    }
    httpd_resp_send_chunk(req, buf, 0);
    file.close();
  } else {
    httpd_resp_send_404(req);
    Storage::end();
  }
}