//---------------------------------
// 2020/8/1

#include "TimelyTaskManager.h"

// n : 登録できるタスクの最大数。IDは 0 - n-1 の範囲で指定
// source : Timeライブラリから時刻を取得 (true), millisから起動からの時間を取得 (false)
TimelyTaskManager::TimelyTaskManager(uint8_t n) {
  taskNum = n;
  if (n > 0) {
    taskList = new Task[n];
  }
  setTimeZone(-3600 * 9, 0);  // タイムゾーンを合わせる ネットワークを初期化せずにconfigTimeを呼ぶとクラッシュするため
}

TimelyTaskManager::~TimelyTaskManager() {
  if (taskNum > 0) {
    delete[] taskList;
  }
}

// タイムゾーンを設定
void TimelyTaskManager::setTimeZone(long offset, int daylight) {
  char cst[17] = {0};
  char cdt[17] = "DST";
  char tz[33] = {0};

  if (offset % 3600) {
    sprintf(cst, "UTC%ld:%02u:%02u", offset / 3600, (unsigned int)abs((offset % 3600) / 60), (unsigned int)abs(offset % 60));
  } else {
    sprintf(cst, "UTC%ld", offset / 3600);
  }
  if (daylight != 3600) {
    long tz_dst = offset - daylight;
    if (tz_dst % 3600) {
      sprintf(cdt, "DST%ld:%02u:%02u", tz_dst / 3600, (unsigned int)abs((tz_dst % 3600) / 60), (unsigned int)abs(tz_dst % 60));
    } else {
      sprintf(cdt, "DST%ld", tz_dst / 3600);
    }
  }
  sprintf(tz, "%s%s", cst, cdt);
  setenv("TZ", tz, 1);
  tzset();
}

// 時刻をNTPから取得
// ESP32 coreのバージョンによっては、ネットワーク初期化前に呼ぶとクラッシュする
void TimelyTaskManager::config() { configTime(3600 * 9, 0, "ntp.nict.jp"); }

// 毎日実行するタスクを登録
void TimelyTaskManager::setDailyTask(uint8_t id, uint8_t setHour, uint8_t setMinute, std::function<void()> task) {
  if (id < taskNum) {
    taskList[id].set = taskDaily;
    taskList[id].hour = setHour;
    taskList[id].minute = setMinute;
    taskList[id].second = 0;
    taskList[id].task = task;
    taskList[id].lastUpdateDay = 0;

    StandaloneTime satime;
    getSaTime(&satime);
    if (satime.hour > setHour || (satime.hour == setHour && satime.minute > setMinute)) {
      taskList[id].lastUpdateDay = satime.day;
    }
  }
}

// 毎時間実行するタスクを登録
void TimelyTaskManager::setHourlyTask(uint8_t id, uint8_t setMinute, std::function<void()> task) {
  if (id < taskNum) {
    taskList[id].set = taskHourly;
    taskList[id].hour = 0;
    taskList[id].minute = setMinute;
    taskList[id].second = 0;
    taskList[id].task = task;
    taskList[id].lastUpdateHour = 0;

    StandaloneTime satime;
    getSaTime(&satime);
    if (satime.minute > setMinute) {
      taskList[id].lastUpdateHour = satime.hour;
    }
  }
}

// 毎分実行するタスクを登録
void TimelyTaskManager::setMinutelyTask(uint8_t id, uint8_t setSecond, std::function<void()> task) {
  if (id < taskNum) {
    taskList[id].set = taskMinutely;
    taskList[id].hour = 0;
    taskList[id].minute = 0;
    taskList[id].second = setSecond;
    taskList[id].task = task;
    taskList[id].lastUpdateMinute = 0;

    StandaloneTime satime;
    getSaTime(&satime);
    if (satime.second > setSecond) {
      taskList[id].lastUpdateMinute = satime.minute;
    }
  }
}

// 毎秒実行するタスクを登録
void TimelyTaskManager::setSecondlyTask(uint8_t id, uint8_t interval, std::function<void()> task) {
  if (id < taskNum) {
    taskList[id].set = taskSecondly;
    taskList[id].hour = 0;
    taskList[id].minute = 0;
    taskList[id].second = interval;
    taskList[id].task = task;
    taskList[id].lastUpdateSecond = 0xFF;
  }
}

// 1回実行するタスクを登録
void TimelyTaskManager::setOneTimeTask(uint8_t id, std::function<void()> task) {
  if (id < taskNum) {
    taskList[id].set = taskOneTime;
    taskList[id].hour = 0;
    taskList[id].minute = 0;
    taskList[id].second = 0;
    taskList[id].task = task;
    taskList[id].lastUpdateSecond = 0xFF;
  }
}

// タスクを削除
void TimelyTaskManager::removeTask(uint8_t id) {
  if (id < taskNum) {
    taskList[id].set = taskNotSet;
    taskList[id].task = NULL;
  }
}

// 予定の時刻を過ぎているタスクを実行
void TimelyTaskManager::updateTask() {
  StandaloneTime satime;
  getSaTime(&satime);

  for (uint8_t i = 0; i < taskNum; i++) {
    switch (taskList[i].set) {
      case taskDaily:
        if (!(satime.day == taskList[i].lastUpdateDay) && (satime.hour > taskList[i].hour || (satime.hour == taskList[i].hour && satime.minute >= taskList[i].minute))) {
          taskList[i].lastUpdateDay = satime.day;
          taskList[i].task();
        }
        break;

      case taskHourly:
        if (!(satime.hour == taskList[i].lastUpdateHour) && satime.minute >= taskList[i].minute) {
          taskList[i].lastUpdateHour = satime.hour;
          taskList[i].task();
        }
        break;

      case taskMinutely:
        if (!(satime.minute == taskList[i].lastUpdateMinute) && satime.second >= taskList[i].second) {
          taskList[i].lastUpdateMinute = satime.minute;
          taskList[i].task();
        }
        break;

      case taskSecondly: {
        // 1st time
        if (taskList[i].lastUpdateSecond == 0xFF) {
          taskList[i].lastUpdateSecond = satime.second;
          taskList[i].task();
        }

        uint8_t current = satime.second;
        if (current < taskList[i].lastUpdateSecond) {
          current += 60;
        }

        if (current >= taskList[i].lastUpdateSecond + taskList[i].second) {
          taskList[i].lastUpdateSecond = satime.second;
          taskList[i].task();
        }
        break;
      }

      case taskOneTime:
        if (taskList[i].lastUpdateSecond == 0xFF) {
          taskList[i].lastUpdateSecond = satime.second;
          taskList[i].task();
        }
        break;
    }
  }
}

bool TimelyTaskManager::getSaTime(StandaloneTime* satime) {
  tm timeinfo;
  if (timeSource) {
    getLocalTime(&timeinfo);
    satime->second = timeinfo.tm_sec;
    satime->minute = timeinfo.tm_min;
    satime->hour = timeinfo.tm_hour;
    satime->day = timeinfo.tm_mday;
    satime->month = timeinfo.tm_mon + 1;
    satime->year = timeinfo.tm_year + 1900;
    return true;
  } else {
    uint32_t sec = millis() / 1000;
    uint32_t min = sec / 60;
    satime->second = sec % 60;
    uint32_t hour = min / 60;
    satime->minute = min % 60;
    satime->day = hour / 24;
    satime->hour = hour % 24;
    return false;
  }
}

String TimelyTaskManager::getTimeStr() { return getTimeStr(formatMinute); }

String TimelyTaskManager::getTimeStr(uint8_t format) {
  StandaloneTime satime;
  char c[32];

  if (getSaTime(&satime)) {
    switch (format) {
      case formatSecond:
        sprintf(c, "%04d/%02d/%02d %02d:%02d:%02d", satime.year, satime.month, satime.day, satime.hour, satime.minute, satime.second);
        break;
      case formatMinute:
        sprintf(c, "%04d/%02d/%02d %02d:%02d", satime.year, satime.month, satime.day, satime.hour, satime.minute);
        break;
      case formatDay:
      default:
        sprintf(c, "%04d/%02d/%02d", satime.year, satime.month, satime.day);
        break;
    }
  } else {
    sprintf(c, "%3d %02d:%02d:%02d", satime.day, satime.hour, satime.minute, satime.second);
  }

  return String(c);
}

String TimelyTaskManager::getTimeFileName(uint8_t format) {
  StandaloneTime satime;
  char c[32];

  if (getSaTime(&satime)) {
    switch (format) {
      case formatSecond:
        sprintf(c, "%02d-%02d%02d-%02d%02d-%02d", satime.year, satime.month, satime.day, satime.hour, satime.minute, satime.second);
        break;
      case formatMinute:
        sprintf(c, "%02d-%02d%02d-%02d%02d", satime.year, satime.month, satime.day, satime.hour, satime.minute);
        break;
      case formatDay:
      default:
        sprintf(c, "%02d-%02d%02d", satime.year, satime.month, satime.day);
        break;
    }
  } else {
    sprintf(c, "%03d-%02d-%02d%02d", satime.day, satime.hour, satime.minute, satime.second);
  }

  return String(c);
}
