
#include <WiFi.h>

#include "INA219I2C.h"
#include "LCDAQM.h"
#include "Storage.h"
#include "TimelyTaskManager.h"
#include "WebPageHandler.h"

//---------------------------------
// 定数

const String wifiSSID = "yourssid";      // WiFiのSSIDを入力
const String wifiPass = "yourpassowrd";  // WiFiのパスワードを入力
// #define staticIP 192, 168, 1, 100       // IPを固定する場合はコメントを外してIPを記入する(カンマ,で区切る)

// バッテリー充放電のカスタマイズ
float vBatTarget = 13.8;      // 最大充電電圧の目標値
const float vBatG1 = 0.1;     // vBatTarget +/- vBatG1を外れたらPWMを調整する
const float vBatG2 = 0.5;     // vBatTarget +/- vBatG2を外れたらPWMを大きく調整する
float cBatTarget = 1.0;       // 最大充電電流の目標値 (ソーラーからの入力電流 - 負荷への出力電流)
const float cBatG1 = 0.1;     // cBatTarget +/- cBatG1を外れたらPWMを調整する
const float cBatG2 = 0.5;     // cBatTarget +/- cBatG2を外れたらPWMを大きく調整する
float vLoadCut = 10.8;        // バッテリー電圧がこの値を下回ったら負荷を切断する(自動の場合)
float vLoadRecover = 12.0;    // バッテリー電圧がこの値を上回ったら負荷の切断を解除する(自動の場合)
float vChargerSleep = 8.0;    // ソーラー入力電圧がこの値を下回るとスリープに入る
float vChargerWakeup = 11.0;  // ソーラー入力電圧がこの値を上回るとスリープを解除

// ピン番号
const uint8_t pinSW2 = 0;
const uint8_t pinSW3 = 4;
const uint8_t pinSW4 = 16;
const uint8_t pinSW5 = 5;
const uint8_t pinSDCS = 17;
const uint8_t pinSDA = 21;
const uint8_t pinSCL = 22;
const uint8_t pinPWM = 32;
const uint8_t pinLE = 13;
const uint8_t pinLoad = 33;
const uint8_t pinLED1 = 25;
const uint8_t pinLED2 = 26;
const uint8_t pinLED3 = 27;
const uint8_t pinLED4 = 14;
const uint8_t pinADC = 36;

// メニュー番号
const uint8_t menuBatteryVC = 0;
const uint8_t menuBatteryP = 1;
const uint8_t menuChargerV = 2;
const uint8_t menuLog = 3;
const uint8_t menuLoad = 4;
const uint8_t menuSleep = 5;
const uint8_t lastMenuId = 5;

// TimelyTaskManagerで管理するためのタスクID
//  9はWebPageHandlerで使用する
const uint8_t idDisplayUpdate = 0;
const uint8_t idSleepControl = 1;
const uint8_t idLoadControl = 2;
const uint8_t idLog = 3;

//---------------------------------
// 変数

TimelyTaskManager timelyTaskManager(10);  // 一定時間おきのタスクスケジューラー
RTC_DATA_ATTR bool timeSource = true;     // TimelyTaskManagerの時刻の取得元

INA219I2C ina219(pinSDA, pinSCL);  // INA219電圧、電流測定IC
LCDAQM lcd(pinSDA, pinSCL);        // LCD

uint16_t pwm = 0;                     // PWMデューティー比 0-256
RTC_DATA_ATTR bool load = true;       // true:自動 false:OFF
RTC_DATA_ATTR bool loadCut = false;   // ture:バッテリー電圧が下がって負荷切断状態
RTC_DATA_ATTR bool sleepMode = true;  // true:自動 false:OFF
RTC_DATA_ATTR bool logMode = false;   // true:測定値をログ中

// LEDと負荷接続の設定
//  Deep sleepから復帰した際も値を保持する変数を使う
//  デバイスに反映させるには値を変更した後にchangeStateを呼ぶようにする
RTC_DATA_ATTR uint8_t stateLED1 = HIGH;
RTC_DATA_ATTR uint8_t stateLED2 = HIGH;
RTC_DATA_ATTR uint8_t stateLED3 = HIGH;
RTC_DATA_ATTR uint8_t stateLED4 = HIGH;
RTC_DATA_ATTR uint8_t stateLoad = HIGH;

bool wakeUpReset = false;      // 電源ONかリセットで起動した場合にtrueにする
bool wakeUpTimer = false;      // タイマーでDeep sleepから復帰した場合はtrueにする
uint8_t menu = menuBatteryVC;  // 現在のメニュー番号
bool changeMode = false;       // SW3を押して値変更モードに入っている場合はtrueにする
bool tmpSetting = false;       // 値変更モードでSW4, 5を押してLCD上で値が変わっている場合はtrueにする
                               // SW3を再度押して決定して実際に反映される

// swReadでスイッチが押して離されたか判定するための変数
uint8_t prevSW2 = 1;
uint8_t prevSW3 = 1;
uint8_t prevSW4 = 1;
uint8_t prevSW5 = 1;

// ロギング
RTC_DATA_ATTR char logPath[128] = "/default.log";  // ログファイル名。実際には測定開始時間のファイル名が生成される。
String logDir = "/log";                            // ログファイルディレクトリ

//---------------------------------
// 関数

void setup() {
  // シリアル通信初期化
  Serial.begin(115200);
  while (!Serial) delay(10);

  // ピンの初期化
  digitalWrite(pinLE, LOW);     // LOW = ラッチ無効
  digitalWrite(pinLoad, HIGH);  // HIGH = OFF
  digitalWrite(pinLED1, HIGH);  // HIGH = OFF
  digitalWrite(pinLED2, HIGH);  // HIGH = OFF
  digitalWrite(pinLED3, HIGH);  // HIGH = OFF
  digitalWrite(pinLED4, HIGH);  // HIGH = OFF
  pinMode(pinLE, OUTPUT);
  pinMode(pinLoad, OUTPUT);
  pinMode(pinLED1, OUTPUT);
  pinMode(pinLED2, OUTPUT);
  pinMode(pinLED3, OUTPUT);
  pinMode(pinLED4, OUTPUT);
  pinMode(pinSW2, PULLUP);
  pinMode(pinSW3, PULLUP);
  pinMode(pinSW4, PULLUP);
  pinMode(pinSW5, PULLUP);
  ledcSetup(0, 500, 8);      // PWM 500Hzで使用
  ledcAttachPin(pinPWM, 0);  // PWMピン

  timelyTaskManager.timeSource = timeSource;

  // 起動理由の取得
  esp_sleep_wakeup_cause_t wakeup_reason;
  wakeup_reason = esp_sleep_get_wakeup_cause();

  switch (wakeup_reason) {
    default:  // 電源ONかリセット
      wakeUpReset = true;
      powerOnInit();
      wifiControl();
      loadControl();
      sleepControl();
      break;
    case ESP_SLEEP_WAKEUP_EXT0:  // SW3でDeep sleepから復帰
      lcd.clear();
      sleepMode = false;  // Deep sleepに入らないようにする
      ina219.vbus(true);
      ina219.current(true);
      loadControl();
      sleepControl();
      wifiControl();
      break;
    case ESP_SLEEP_WAKEUP_TIMER:  // タイマーでDeep sleepから復帰
      wakeUpTimer = true;
      ina219.vbus(true);
      ina219.current(true);
      lcd.clear();
      if (logMode) {
        oneTimeLog();
      }
      loadControl();
      sleepControl();
      wifiControl();
      break;
  }

  // TimelyTaskManagerにタスクを登録
  timelyTaskManager.setSecondlyTask(idDisplayUpdate, 1, displayUpdate);
  timelyTaskManager.setMinutelyTask(idSleepControl, 5, sleepControl);
  timelyTaskManager.setMinutelyTask(idLoadControl, 10, loadControl);
  if (logMode) {
    timelyTaskManager.setMinutelyTask(idLog, 0, oneTimeLog);
  }

  // LCDに簡単なボタンの機能を表示
  lcd.clear();
  lcd.printStr("2:PGMDL");
  lcd.secLine();
  lcd.printStr("3:SET/WU");
  delay(2000);
  lcd.clear();
  lcd.printStr("4:+");
  lcd.secLine();
  lcd.printStr("5:-");
  delay(2000);

  // メニューを初期化
  menu = lastMenuId;
  swEvent(2);
}

void loop() {
  timelyTaskManager.updateTask();  // 登録されているタスクを処理
  pwmControl();                    // PWM調整
  swEvent(swRead());               // スイッチのが押されたら処理
  delay(10);
}

// 電源投入、リセット時に呼ばれる初期化処理
void powerOnInit() {
  // SW3を押しながら起動した場合はsleepに入らないようにする
  if (digitalRead(pinSW3) == 0) {
    sleepMode = false;
  }

  Storage::opLog("\n\n-----------------------\n");
  Storage::opLog("Program start\n");

  btStop();

  // LCD初期化
  lcd.init();

  if (!sleepMode) {
    lcd.printStr("SLEEP");
    lcd.secLine();
    lcd.printStr("DISABLED");
    delay(2000);
    lcd.clear();
  }

  // INA219初期化
  if (!ina219.init()) {
    Storage::opLog("INA219 Error\n");
    lcd.printStr("INA219");
    lcd.secLine();
    lcd.printStr("ERROR");

    // INA219との通信に失敗したら動作停止
    while (true) {
      delay(1000);
    }
  }
}

// WebPageHandlerから呼び出せる関数
// この例では使用しない
void webPageHandlerCallBack(uint8_t action) {
  // switch(action){
  //  ...
  //}
}

// stateXXX変数の値をラッチICにセットする
// ESP32がスリープ、リセットしても保持される
void changeState() {
  digitalWrite(pinLED1, stateLED1);
  digitalWrite(pinLED2, stateLED2);
  digitalWrite(pinLED3, stateLED3);
  digitalWrite(pinLED4, stateLED4);
  digitalWrite(pinLoad, stateLoad);
  digitalWrite(pinLE, HIGH);
  delayMicroseconds(50);
  digitalWrite(pinLE, LOW);
}

// ソーラ入力電圧をADCで測定
float vcharge() {
  float vcrg = 3.3 * 115 / 15 * analogRead(pinADC) / 4095;
  if (vcrg < 0.1) {
    return 0;
  }
  return vcrg + 1.15;  // 実測値似合わせるための補正
}

// ソーラ入力電圧を文字列で返す
String vchargeStr() {
  char c[8];
  dtostrf(vcharge(), 3, 2, c);
  return String(c);
}

// バッテリーの電圧、電流を測定し、PWMを調整する
void pwmControl() {
  float vBat = ina219.vbus(true);
  float cBat = ina219.current(true);
  float vCharge = vcharge();

  // 充電できるだけの電圧がソーラーから入力されていない場合は止める
  if (vCharge < vChargerWakeup) {
    pwm = 0;
    ledcWrite(0, pwm);
    return;
  }

  // 電圧、電流値に応じてpwm変数を変動させる
  if (vBat > vBatTarget + vBatG2 || cBat > cBatTarget + cBatG2) {
    if (pwm > 0 && pwm <= 50) {
      pwm -= 1;
    } else if (pwm > 50) {
      pwm -= 5;
    }
  } else if (vBat > vBatTarget + vBatG1 || cBat > cBatTarget + cBatG1) {
    if (pwm >= 1) {
      pwm -= 1;
    }
  } else if (vBat < vBatTarget - vBatG2 && cBat < cBatTarget - cBatG2) {
    if (pwm >= 246) {
      pwm = 256;
    } else if (pwm <= 50) {
      pwm += 1;
    } else {
      pwm += 5;
    }
  } else if (vBat < vBatTarget - vBatG1 && cBat < cBatTarget - cBatG1) {
    if (pwm <= 255) {
      pwm += 1;
    }
  }

  // 実際のPWMに反映させる
  ledcWrite(0, pwm);
}

// ソーラー入力電圧を読み取り、スリープに入るか判定
void sleepControl() {
  float vCharge = vcharge();
  float vth = vChargerSleep;  // 通常動作中にスリープに入る電圧のしきい値

  if (wakeUpTimer) {
    vth = vChargerWakeup;  // スリープ状態中にスリープから抜ける電圧のしきい値
  }

  // スリープに入るか、継続する場合
  if (vCharge < vth && sleepMode) {
    // Green LED OFF & Yellow LED ON
    stateLED1 = HIGH;
    stateLED2 = LOW;
    changeState();

    lcd.clear();
    lcd.printStr(ina219.vbusStr(false).c_str());
    lcd.printStr("V");
    lcd.secLine();

    // Deep sleepを継続する場合は動作ログに記録しない
    if (!wakeUpTimer) {
      Storage::opLog(timelyTaskManager.getTimeStr() + " Low VCHARGE. Start deep sleep\n");
    }
    lcd.printStr("SLEEP");
    esp_sleep_enable_ext0_wakeup(GPIO_NUM_4, 0);  // SW3が押されたらDeep sleepから復帰
    ESP.deepSleep(15 * 60 * 1000000);             // Deep sleepに入る. 15分経過で復帰
  }

  // スリープに入らない場合
  // Green LED ON & Yellow LED OFF
  stateLED1 = LOW;
  stateLED2 = HIGH;
  changeState();

  wakeUpTimer = false;  // スリープに入らなかった場合はfalseに直しておく
}

// バッテリーの電圧を測定し、負荷の接続、切断制御をする
void loadControl() {
  // 負荷制御が自動
  if (load) {
    float vBat = ina219.vbus(false);

    // 電圧が下がって負荷切断状態のとき
    if (loadCut) {
      if (vBat > vLoadRecover) {
        stateLoad = LOW;  // 負荷接続
        stateLED3 = HIGH;
        loadCut = false;
        Storage::opLog(timelyTaskManager.getTimeStr() + " VBATTERY recoverd. Connected the load.\n");
      } else {
        stateLoad = HIGH;  // 負荷切断
        stateLED3 = LOW;
      }
    }
    // 負荷接続可能状態のとき
    else {
      if (vBat < vLoadCut) {
        stateLoad = HIGH;  // 負荷切断
        stateLED3 = LOW;
        loadCut = true;
        Storage::opLog(timelyTaskManager.getTimeStr() + " Low VBATTERY. Cut off the load.\n");
      } else {
        stateLoad = LOW;  // 負荷接続
        stateLED3 = HIGH;
      }
    }
  } else {
    stateLoad = HIGH;  // 負荷切断
    stateLED3 = HIGH;
  }
  changeState();
}

// WiFiを設定、開始する
void wifiControl() {
  if (wakeUpReset || (!timeSource)) {
    Storage::opLog("\nWiFi");
  } else {
    Storage::opLog(timelyTaskManager.getTimeStr() + " WiFi");
  }

  lcd.printStr("WIFI..");
  lcd.secLine();

  WiFi.mode(WIFI_STA);
  WiFi.begin(wifiSSID.c_str(), wifiPass.c_str());

  // 接続されるまで待機
  for (uint16_t i = 0; i < 20; i++) {
    // 接続成功
    if (WiFi.status() == WL_CONNECTED) {
#ifdef staticIP
      // 固定IPを使う場合
      WiFi.config(IPAddress(staticIP), WiFi.gatewayIP(), WiFi.subnetMask(), WiFi.dnsIP());
#endif
      Storage::opLog(" Connected IP : " + WiFi.localIP().toString() + "\n");
      lcd.printStr("CONNECTD");
      delay(1000);
      break;
    }
    // 環境、タイミングによって1回目のWiFi接続がうまくいかないことがある
    else if (i == 9) {
      WiFi.begin(wifiSSID.c_str(), wifiPass.c_str());
      delay(1000);
    }
    // 接続失敗
    else if (i == 19) {
      Storage::opLog(" Not Found\n");
      lcd.printStr("NOTFOUND");
      WiFi.mode(WIFI_OFF);
      timeSource = false;  // TimelyTaskManagerの時刻をmillisから取得
      timelyTaskManager.timeSource = false;
      delay(1000);
      return;
    }
    delay(500);
  }

  // TimelyTaskManagerの時間をNTPから取得
  timeSource = true;
  timelyTaskManager.timeSource = true;
  timelyTaskManager.config();
  Storage::opLog(timelyTaskManager.getTimeStr() + " Time Configured\n");

  // Web Server
  WebPageHandler::startServer(webPageHandlerCallBack);
  Storage::opLog(timelyTaskManager.getTimeStr() + " Web Server Started\n");
}

// バッテリーの電圧などを測定してログファイルに記録(1回分)
void oneTimeLog() {
  String timeStr = timelyTaskManager.getTimeStr(TimelyTaskManager::formatSecond);
  if (!Storage::exists(logDir)) {
    Storage::mkdir(logDir);
  }
  bool exists = Storage::exists(logPath);
  File file = Storage::open(logPath, FILE_APPEND);
  if (file) {
    if (!exists) {
      file.print("Time,Battery V[V],Battery I[A],Battery P[W],Input V[V]\n");
    }
    file.print(timeStr.c_str());
    file.print(",");
    file.print(ina219.vbusStr(false).c_str());
    file.print(",");
    file.print(ina219.currentStr(false).c_str());
    file.print(",");
    char p[8];
    dtostrf(ina219.vbus(false) * ina219.current(false), 1, 2, p);
    file.print(p);
    file.print(",");
    file.print(vchargeStr());
    file.print("\n");
    file.close();
  }
  Storage::end();
}

// 測定値のロギングを開始する
void startLog() {
  logMode = true;
  Storage::opLog(timelyTaskManager.getTimeStr() + " Start Log Mode\n");
  String s;
  if (logDir.endsWith("/")) {
    s = logDir + timelyTaskManager.getTimeFileName(TimelyTaskManager::formatSecond) + ".csv";
  } else {
    s = logDir + "/" + timelyTaskManager.getTimeFileName(TimelyTaskManager::formatSecond) + ".csv";
  }
  strcpy(logPath, s.c_str());
  timelyTaskManager.setMinutelyTask(idLog, 0, oneTimeLog);
}

// 測定値のロギングを停止する
void stopLog() {
  logMode = false;
  timelyTaskManager.removeTask(idLog);
  Storage::opLog(timelyTaskManager.getTimeStr() + " Finish Log Mode\n");
}

// メニューの状態に合わせた内容をLCDに表示
// 一定時間ごとに呼び出してLCDを更新するようにする
void displayUpdate() {
  switch (menu) {
    case menuBatteryVC:
      displayBatteryVC();
      break;
    case menuBatteryP:
      displayBatteryP();
      break;
    case menuChargerV:
      displayChargeV();
      break;
    case menuLog:
      lcd.clear();
      lcd.printStr("LOG");
      lcd.secLine();
      if (changeMode) {
        if (tmpSetting) {
          lcd.printStr("(ON)");
        } else {
          lcd.printStr("(OFF)");
        }
      } else {
        if (logMode) {
          lcd.printStr("ON");
        } else {
          lcd.printStr("OFF");
        }
      }
      break;
    case menuLoad:
      lcd.clear();
      lcd.printStr("LOAD");
      lcd.secLine();
      if (changeMode) {
        if (tmpSetting) {
          lcd.printStr("(AUTO)");
        } else {
          lcd.printStr("(OFF)");
        }
      } else {
        if (load) {
          lcd.printStr("AUTO");
        } else {
          lcd.printStr("OFF");
        }
      }
      break;
    case menuSleep:
      lcd.clear();
      lcd.printStr("SLEEP");
      lcd.secLine();
      if (changeMode) {
        if (tmpSetting) {
          lcd.printStr("(AUTO)");
        } else {
          lcd.printStr("(OFF)");
        }
      } else {
        if (sleepMode) {
          lcd.printStr("AUTO");
        } else {
          lcd.printStr("OFF");
        }
      }
      break;
  }
}

// LCDにバッテリーの電圧と充電(放電)電流を表示
void displayBatteryVC() {
  lcd.clear();
  lcd.printStr(ina219.vbusStr(false).c_str());
  lcd.printStr("V");
  lcd.secLine();
  lcd.printStr(ina219.currentStr(false).c_str());
  lcd.printStr("A");
}

// LCDにバッテリーの充電(放電)電力を表示
void displayBatteryP() {
  char p[8];
  dtostrf(ina219.vbus(false) * ina219.current(false), 1, 2, p);
  lcd.clear();
  lcd.printStr("BAT P");
  lcd.secLine();
  lcd.printStr(p);
  lcd.printStr("W");
}

// LCDにソーラー入力電圧を表示
void displayChargeV() {
  lcd.clear();
  lcd.printStr("CRG V");
  lcd.secLine();
  lcd.printStr(vchargeStr().c_str());
  lcd.printStr("V");
}

// スイッチの状態を監視して、押して離す動作を検出したらその番号を返す
// どのスイッチも押されていなければ0を返す
uint8_t swRead() {
  // SW2
  if (digitalRead(pinSW2) == 0) {
    prevSW2 = 0;
  } else {
    if (prevSW2 == 0) {
      prevSW2 = 1;
      return 2;
    }
  }

  // SW3
  if (digitalRead(pinSW3) == 0) {
    prevSW3 = 0;
  } else {
    if (prevSW3 == 0) {
      prevSW3 = 1;
      return 3;
    }
  }

  // SW4
  if (digitalRead(pinSW4) == 0) {
    prevSW4 = 0;
  } else {
    if (prevSW4 == 0) {
      prevSW4 = 1;
      return 4;
    }
  }

  // SW5
  if (digitalRead(pinSW5) == 0) {
    prevSW5 = 0;
  } else {
    if (prevSW5 == 0) {
      prevSW5 = 1;
      return 5;
    }
  }
  return 0;
}

// SWが押された際にする処理
//  sw : スイッチ番号
void swEvent(uint8_t sw) {
  switch (sw) {
    // 無効な番号の場合は何もしない
    default:
      return;

    // SW2が押された場合
    case 2:
      if (changeMode) {
        return;
      }

      menu++;
      if (menu > lastMenuId) {
        lcd.clear();
        lcd.printStr("BAT V");
        lcd.secLine();
        lcd.printStr("BAT C");
        delay(1000);
        menu = 0;
      }
      displayUpdate();
      break;

    // SW3が押された場合
    case 3:
      switch (menu) {
        case menuBatteryVC:
          break;
        case menuBatteryP:
          break;
        case menuChargerV:
          break;
        case menuLog:
          if (changeMode) {
            changeMode = false;
            if (tmpSetting && !logMode) {
              startLog();
            } else if (!tmpSetting && logMode) {
              stopLog();
            }
            displayUpdate();
          } else {
            changeMode = true;
            tmpSetting = logMode;
            displayUpdate();
          }
          break;

        case menuLoad:
          if (changeMode) {
            changeMode = false;
            load = tmpSetting;
            displayUpdate();
            loadControl();
          } else {
            changeMode = true;
            tmpSetting = load;
            displayUpdate();
          }
          break;

        case menuSleep:
          if (changeMode) {
            changeMode = false;
            sleepMode = tmpSetting;
            displayUpdate();
            sleepControl();
          } else {
            changeMode = true;
            tmpSetting = sleepMode;
            displayUpdate();
          }
          break;
      }
      break;

    // SW4が押された場合
    case 4:
      if (!changeMode) {
        menu++;
        if (menu > lastMenuId) {
          lcd.clear();
          lcd.printStr("BAT V");
          lcd.secLine();
          lcd.printStr("BAT C");
          delay(1000);
          menu = 0;
        }
        displayUpdate();
      } else {
        switch (menu) {
          case menuLog:
          case menuLoad:
          case menuSleep:
            if (changeMode) {
              tmpSetting = !tmpSetting;
              displayUpdate();
            }
            break;
        }
      }
      break;

    // SW5が押された場合
    case 5:
      if (!changeMode) {
        if (menu == 1) {
          lcd.clear();
          lcd.printStr("BAT V");
          lcd.secLine();
          lcd.printStr("BAT C");
          delay(1000);
          menu = 0;
        } else if (menu == 0) {
          menu = lastMenuId;
        } else {
          menu--;
        }
        displayUpdate();
      } else {
        switch (menu) {
          case menuLog:
          case menuLoad:
          case menuSleep:
            if (changeMode) {
              tmpSetting = !tmpSetting;
              displayUpdate();
            }
            break;
        }
      }
      break;
  }
}
