//---------------------------------
// 2020/9/21

#include "INA219I2C.h"

INA219I2C::INA219I2C(uint8_t sda, uint8_t scl) {
  pinSDA = sda;
  pinSCL = scl;
}

// Read 1word from register
uint16_t INA219I2C::readAddress(uint8_t addr) {
  Wire.begin(pinSDA, pinSCL);
  Wire.beginTransmission(i2cAddr);
  Wire.write(addr);
  Wire.endTransmission();
  Wire.requestFrom((uint8_t)i2cAddr, (uint8_t)2);
  uint8_t high = Wire.read();
  uint8_t low = Wire.read();
  return (high << 8) | low;
}

// Write 1word to register
void INA219I2C::writeAddress(uint8_t addr, uint16_t data) {
  Wire.begin(pinSDA, pinSCL);
  Wire.beginTransmission(i2cAddr);
  Wire.write(addr);
  Wire.write(data >> 8);
  Wire.write(data & 0xFF);
  Wire.endTransmission();
}

bool INA219I2C::init() {
  writeAddress(0x0, 0x399F);
  writeAddress(0x5, cal);
  delay(1);
  if (readAddress(0x0) == 0x399F) {
    vbus(true);
    current(true);
    power(true);
    return true;
  }

  return false;
}

// Return measured voltage
//  meas : If false, return last measured value. No I2C transfer.
float INA219I2C::vbus(bool meas) {
  if (meas) {
    lastVbus = (float)((readAddress(0x2) >> 3) << 5) / 8000;  // Vbus register x 32V range / 8000;
  }
  return lastVbus;
}

// Return measured voltage string
//  meas : If false, return last measured value. No I2C transfer.
String INA219I2C::vbusStr(bool meas) {
  char c[8];
  dtostrf(vbus(meas), 3, 2, c);
  return String(c);
}

// Return measured current
//  meas : If false, return last measured value. No I2C transfer.
float INA219I2C::current(bool meas) {
  if (meas) {
    float m = 0.0;
    for(uint16_t i=0; i<100; i++){
      uint16_t regUint = readAddress(0x4);
      int16_t regInt;
      if (regUint <= 32767) {
        regInt = regUint;
      } else {
        regInt = regUint - 65536;
      }
      m += (float)regInt * currentLSB / 1000;  // A
      delayMicroseconds(20);
    }
    lastCurrent = m/100.0;
  }
  if (lastCurrent < 0.01 && lastCurrent > -0.01) {
    return 0;
  }
  return lastCurrent;
}

// Return measured current string
//  meas : If false, return last measured value. No I2C transfer.
String INA219I2C::currentStr(bool meas) {
  char c[8];
  dtostrf(current(meas), 1, 2, c);
  return String(c);
}

// Return measured power
//  meas : If false, return last measured value. No I2C transfer.
float INA219I2C::power(bool meas) {
  if (meas) {
    lastPower = (float)readAddress(0x3) * powerLSB / 10;  // W
  }
  return lastPower;
}

// Return measured power string
//  meas : If false, return last measured value. No I2C transfer.
String INA219I2C::powerStr(bool meas) {
  char c[8];
  dtostrf(power(meas), 1, 1, c);
  return String(c);
}
